/*
 * Decompiled with CFR 0.152.
 */
package dev.ninesliced.managers;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.annotations.SerializedName;
import com.hypixel.hytale.component.Holder;
import com.hypixel.hytale.protocol.Transform;
import com.hypixel.hytale.protocol.packets.worldmap.ContextMenuItem;
import com.hypixel.hytale.protocol.packets.worldmap.MapMarker;
import com.hypixel.hytale.server.core.entity.entities.Player;
import com.hypixel.hytale.server.core.entity.entities.player.data.PlayerWorldData;
import com.hypixel.hytale.server.core.universe.PlayerRef;
import com.hypixel.hytale.server.core.universe.Universe;
import com.hypixel.hytale.server.core.universe.world.World;
import com.hypixel.hytale.server.core.util.PositionUtil;
import dev.ninesliced.configs.BetterMapConfig;
import dev.ninesliced.listeners.ExplorationEventListener;
import dev.ninesliced.utils.PermissionsUtil;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.Reader;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Logger;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

public class WaypointManager {
    private static final Logger LOGGER = Logger.getLogger(WaypointManager.class.getName());
    private static final String GLOBAL_ID_PREFIX = "global_waypoint_";
    private static WaypointPersistence persistence;
    private static final Set<UUID> loadedPlayers;

    private WaypointManager() {
    }

    public static void initialize(@Nonnull Path configDir) {
        persistence = new WaypointPersistence(configDir);
    }

    @Nullable
    public static MapMarker[] getWaypoints(@Nonnull Player player) {
        World world = player.getWorld();
        if (world == null || !ExplorationEventListener.isTrackedWorld(world)) {
            return new MapMarker[0];
        }
        WaypointManager.ensureLoaded(player, world);
        PlayerWorldData perWorldData = player.getPlayerConfigData().getPerWorldData(world.getName());
        MapMarker[] markers = perWorldData.getWorldMapMarkers();
        return markers != null ? markers : new MapMarker[]{};
    }

    public static void addWaypoint(@Nonnull Player player, @Nonnull String name, @Nonnull String icon, @Nonnull Transform transform, boolean global) {
        World world = player.getWorld();
        if (world == null || !ExplorationEventListener.isTrackedWorld(world)) {
            return;
        }
        WaypointManager.ensureLoaded(player, world);
        String markerId = (global ? GLOBAL_ID_PREFIX : "waypoint_") + String.valueOf(UUID.randomUUID());
        MapMarker marker = new MapMarker(markerId, name, WaypointManager.normalizeIcon(icon), transform, WaypointManager.buildContextMenu(player, markerId));
        if (global) {
            WaypointManager.saveGlobalMarker(marker, world, player);
        } else {
            WaypointManager.savePersonalMarker(player, world, marker);
        }
        UUID playerUuid = player.getUuid();
        loadedPlayers.remove(playerUuid);
    }

    public static boolean removeWaypoint(@Nonnull Player player, @Nonnull String idOrName) {
        World world = player.getWorld();
        if (world == null || !ExplorationEventListener.isTrackedWorld(world)) {
            return false;
        }
        WaypointManager.ensureLoaded(player, world);
        MapMarker target = WaypointManager.findWaypoint(player, idOrName);
        if (target == null || target.id == null) {
            return false;
        }
        if (WaypointManager.isGlobalId(target.id)) {
            boolean result = WaypointManager.removeGlobalMarker(target.id, world.getName(), player);
            if (result) {
                UUID playerUuid = player.getUuid();
                loadedPlayers.remove(playerUuid);
                WaypointManager.refreshAllPlayersMarkers(world);
                WaypointManager.ensureLoaded(player, world);
            }
            return result;
        }
        PlayerWorldData perWorldData = player.getPlayerConfigData().getPerWorldData(world.getName());
        MapMarker[] currentMarkers = perWorldData.getWorldMapMarkers();
        if (currentMarkers == null) {
            return false;
        }
        ArrayList<MapMarker> newMarkerList = new ArrayList<MapMarker>();
        boolean found = false;
        for (MapMarker marker : currentMarkers) {
            boolean match;
            boolean bl = match = marker != null && marker.id != null && marker.id.equalsIgnoreCase(target.id);
            if (!match && marker != null && marker.name != null) {
                match = WaypointManager.stripColorTags(marker.name).equalsIgnoreCase(idOrName);
            }
            if (match) {
                found = true;
                continue;
            }
            newMarkerList.add(marker);
        }
        if (found) {
            WaypointManager.persistPersonal(player, world.getName(), newMarkerList);
            UUID playerUuid = player.getUuid();
            loadedPlayers.remove(playerUuid);
            WaypointManager.ensureLoaded(player, world);
        }
        return found;
    }

    public static boolean updateWaypoint(@Nonnull Player player, @Nonnull String id, @Nullable String newName, @Nullable String newIcon, @Nullable Transform newTransform) {
        World world = player.getWorld();
        if (world == null || !ExplorationEventListener.isTrackedWorld(world)) {
            return false;
        }
        WaypointManager.ensureLoaded(player, world);
        if (WaypointManager.isGlobalId(id)) {
            boolean result = WaypointManager.updateGlobalMarker(id, newName, newIcon, newTransform, world.getName(), player);
            if (result) {
                UUID playerUuid = player.getUuid();
                loadedPlayers.remove(playerUuid);
            }
            return result;
        }
        PlayerWorldData perWorldData = player.getPlayerConfigData().getPerWorldData(world.getName());
        MapMarker[] currentMarkers = perWorldData.getWorldMapMarkers();
        if (currentMarkers == null) {
            return false;
        }
        ArrayList<MapMarker> rebuilt = new ArrayList<MapMarker>();
        boolean found = false;
        for (MapMarker m : currentMarkers) {
            if (m != null && m.id != null && m.id.equals(id)) {
                found = true;
                String nameToUse = newName != null ? newName : m.name;
                String iconToUse = newIcon != null ? WaypointManager.normalizeIcon(newIcon) : m.markerImage;
                Transform transformToUse = newTransform != null ? newTransform : m.transform;
                String newId = iconToUse != null && !iconToUse.equals(m.markerImage) ? "waypoint_" + String.valueOf(UUID.randomUUID()) : m.id;
                rebuilt.add(new MapMarker(newId, nameToUse, iconToUse, transformToUse, WaypointManager.buildContextMenu(player, newId)));
                continue;
            }
            rebuilt.add(m);
        }
        if (!found) {
            return false;
        }
        WaypointManager.persistPersonal(player, world.getName(), rebuilt);
        UUID playerUuid = player.getUuid();
        loadedPlayers.remove(playerUuid);
        WaypointManager.ensureLoaded(player, world);
        return true;
    }

    @Nullable
    private static ContextMenuItem[] buildContextMenu(@Nonnull Player player, @Nonnull String markerId) {
        ArrayList<ContextMenuItem> menuItems = new ArrayList<ContextMenuItem>();
        boolean isGlobal = WaypointManager.isGlobalId(markerId);
        menuItems.add(new ContextMenuItem(isGlobal ? "Global Waypoint" : "Personal Waypoint", ""));
        if (PermissionsUtil.canTeleport(player) && BetterMapConfig.getInstance().isAllowWaypointTeleports()) {
            menuItems.add(new ContextMenuItem("Teleport To", "bm waypoint teleport " + markerId));
        }
        if (isGlobal) {
            menuItems.add(new ContextMenuItem("Delete", "bm waypoint removeglobal " + markerId));
        } else {
            menuItems.add(new ContextMenuItem("Delete", "bm waypoint remove " + markerId));
        }
        return menuItems.isEmpty() ? null : menuItems.toArray(new ContextMenuItem[0]);
    }

    public static MapMarker getWaypoint(@Nonnull Player player, @Nonnull String id) {
        MapMarker[] all;
        World world = player.getWorld();
        if (world == null || !ExplorationEventListener.isTrackedWorld(world)) {
            return null;
        }
        WaypointManager.ensureLoaded(player, world);
        for (MapMarker m : all = WaypointManager.getWaypoints(player)) {
            if (m == null || m.id == null || !m.id.equals(id)) continue;
            return m;
        }
        return null;
    }

    public static MapMarker findWaypoint(@Nonnull Player player, @Nonnull String nameOrId) {
        MapMarker[] all;
        World world = player.getWorld();
        if (world == null || !ExplorationEventListener.isTrackedWorld(world)) {
            return null;
        }
        WaypointManager.ensureLoaded(player, world);
        for (MapMarker m : all = WaypointManager.getWaypoints(player)) {
            if (m == null) continue;
            if (m.id != null && m.id.equalsIgnoreCase(nameOrId)) {
                return m;
            }
            if (m.name == null) continue;
            if (m.name.equalsIgnoreCase(nameOrId)) {
                return m;
            }
            if (!WaypointManager.stripColorTags(m.name).equalsIgnoreCase(nameOrId)) continue;
            return m;
        }
        return null;
    }

    private static void refreshPlayerMarkers(@Nonnull Player player) {
        World world = player.getWorld();
        if (world == null || !ExplorationEventListener.isTrackedWorld(world)) {
            return;
        }
        PlayerWorldData perWorldData = player.getPlayerConfigData().getPerWorldData(world.getName());
        MapMarker[] currentMarkers = perWorldData.getWorldMapMarkers();
        ArrayList<MapMarker> personal = new ArrayList<MapMarker>();
        if (currentMarkers != null) {
            for (MapMarker m : currentMarkers) {
                if (m == null || m.id == null || WaypointManager.isGlobalId(m.id)) continue;
                personal.add(m);
            }
        }
        List<MapMarker> globals = WaypointManager.getGlobalMarkers(world.getName(), player);
        ArrayList<MapMarker> combined = new ArrayList<MapMarker>(personal);
        combined.addAll(globals);
        perWorldData.setWorldMapMarkers(combined.toArray(new MapMarker[0]));
    }

    private static void ensureLoaded(@Nonnull Player player, @Nonnull World world) {
        if (persistence == null) {
            return;
        }
        if (!WaypointManager.isTrackedWorld(world)) {
            return;
        }
        UUID uuid = player.getUuid();
        if (!loadedPlayers.add(uuid)) {
            return;
        }
        List<StoredWaypoint> stored = persistence.loadPlayer(uuid, player.getDisplayName(), world.getName());
        ArrayList<MapMarker> markers = new ArrayList<MapMarker>();
        if (!stored.isEmpty()) {
            for (StoredWaypoint waypoint : stored) {
                MapMarker marker = WaypointManager.toMarker(waypoint, player);
                if (marker == null) continue;
                markers.add(marker);
            }
        }
        markers.addAll(WaypointManager.getGlobalMarkers(world.getName(), player));
        PlayerWorldData perWorldData = player.getPlayerConfigData().getPerWorldData(world.getName());
        perWorldData.setWorldMapMarkers(markers.toArray(new MapMarker[0]));
    }

    private static List<MapMarker> getGlobalMarkers(@Nonnull String worldName, @Nonnull Player player) {
        if (persistence == null) {
            return Collections.emptyList();
        }
        List<StoredWaypoint> stored = persistence.loadGlobal();
        if (stored.isEmpty()) {
            return Collections.emptyList();
        }
        ArrayList<MapMarker> markers = new ArrayList<MapMarker>();
        for (StoredWaypoint waypoint : stored) {
            MapMarker marker;
            if (waypoint.world == null || !worldName.equalsIgnoreCase(waypoint.world) || (marker = WaypointManager.toMarker(waypoint, player)) == null) continue;
            markers.add(marker);
        }
        return markers;
    }

    private static void savePersonalMarker(@Nonnull Player player, @Nonnull World world, @Nonnull MapMarker marker) {
        PlayerWorldData perWorldData = player.getPlayerConfigData().getPerWorldData(world.getName());
        MapMarker[] currentMarkers = perWorldData.getWorldMapMarkers();
        ArrayList<MapMarker> newMarkerList = new ArrayList<MapMarker>();
        if (currentMarkers != null) {
            newMarkerList.addAll(Arrays.asList(currentMarkers));
        }
        newMarkerList.add(marker);
        perWorldData.setWorldMapMarkers(newMarkerList.toArray(new MapMarker[0]));
        WaypointManager.persistPersonal(player, world.getName(), newMarkerList);
    }

    private static void persistPersonal(@Nonnull Player player, @Nonnull String worldName, @Nonnull List<MapMarker> markers) {
        if (persistence == null || !ExplorationEventListener.isTrackedWorld(player.getWorld())) {
            return;
        }
        ArrayList<StoredWaypoint> stored = new ArrayList<StoredWaypoint>();
        for (MapMarker marker : markers) {
            StoredWaypoint waypoint;
            if (marker == null || marker.id == null || WaypointManager.isGlobalId(marker.id) || (waypoint = WaypointManager.fromMarker(marker, worldName, player.getDisplayName(), player.getUuid(), false)) == null) continue;
            stored.add(waypoint);
        }
        persistence.savePlayer(player.getUuid(), player.getDisplayName(), worldName, stored);
    }

    private static void saveGlobalMarker(@Nonnull MapMarker marker, @Nonnull World world, @Nonnull Player player) {
        if (persistence == null || !ExplorationEventListener.isTrackedWorld(world)) {
            return;
        }
        List<StoredWaypoint> existing = persistence.loadGlobal();
        ArrayList<StoredWaypoint> mutable = new ArrayList<StoredWaypoint>(existing);
        StoredWaypoint converted = WaypointManager.fromMarker(marker, world.getName(), player.getDisplayName(), player.getUuid(), true);
        if (converted == null) {
            return;
        }
        mutable.add(converted);
        persistence.saveGlobal(mutable);
        world.execute(() -> WaypointManager.refreshAllPlayersMarkers(world));
    }

    public static void refreshAllPlayersMarkers(@Nonnull World world) {
        if (world == null || !ExplorationEventListener.isTrackedWorld(world)) {
            return;
        }
        for (PlayerRef playerRef : world.getPlayerRefs()) {
            Player p;
            Holder holder = playerRef.getHolder();
            if (holder == null || (p = (Player)holder.getComponent(Player.getComponentType())) == null) continue;
            WaypointManager.refreshPlayerMarkers(p);
        }
    }

    private static boolean removeGlobalMarker(@Nonnull String markerId, @Nonnull String worldName, @Nonnull Player player) {
        if (persistence == null) {
            return false;
        }
        List<StoredWaypoint> existing = persistence.loadGlobal();
        ArrayList<StoredWaypoint> updated = new ArrayList<StoredWaypoint>();
        boolean found = false;
        for (StoredWaypoint waypoint : existing) {
            if (waypoint.id != null && waypoint.id.equals(markerId)) {
                found = true;
                continue;
            }
            updated.add(waypoint);
        }
        if (found) {
            persistence.saveGlobal(updated);
            World world = Universe.get().getWorld(worldName);
            if (world != null) {
                world.execute(() -> WaypointManager.refreshAllPlayersMarkers(world));
            }
        }
        return found;
    }

    private static boolean updateGlobalMarker(@Nonnull String markerId, @Nullable String newName, @Nullable String newIcon, @Nullable Transform newTransform, @Nonnull String worldName, @Nonnull Player actor) {
        if (persistence == null) {
            return false;
        }
        List<StoredWaypoint> existing = persistence.loadGlobal();
        boolean found = false;
        ArrayList<StoredWaypoint> rebuilt = new ArrayList<StoredWaypoint>();
        for (StoredWaypoint waypoint : existing) {
            if (waypoint.id != null && waypoint.id.equals(markerId)) {
                found = true;
                String iconToUse = newIcon != null ? WaypointManager.normalizeIcon(newIcon) : waypoint.icon;
                Object newId = waypoint.id;
                if (iconToUse != null && !iconToUse.equals(waypoint.icon)) {
                    newId = GLOBAL_ID_PREFIX + String.valueOf(UUID.randomUUID());
                }
                double x = waypoint.x;
                double y = waypoint.y;
                double z = waypoint.z;
                if (newTransform != null && newTransform.position != null) {
                    x = newTransform.position.x;
                    y = newTransform.position.y;
                    z = newTransform.position.z;
                }
                StoredWaypoint updated = new StoredWaypoint((String)newId, newName != null ? newName : waypoint.name, iconToUse, x, y, z, worldName, true, waypoint.ownerUuid, waypoint.ownerName);
                rebuilt.add(updated);
                continue;
            }
            rebuilt.add(waypoint);
        }
        if (found) {
            persistence.saveGlobal(rebuilt);
            World world = Universe.get().getWorld(worldName);
            if (world != null) {
                world.execute(() -> WaypointManager.refreshAllPlayersMarkers(world));
            }
        }
        return found;
    }

    private static MapMarker toMarker(@Nonnull StoredWaypoint waypoint, @Nonnull Player player) {
        Transform transform = PositionUtil.toTransformPacket((com.hypixel.hytale.math.vector.Transform)new com.hypixel.hytale.math.vector.Transform(waypoint.x, waypoint.y, waypoint.z));
        ContextMenuItem[] menu = WaypointManager.buildContextMenu(player, waypoint.id);
        return new MapMarker(waypoint.id, waypoint.name, WaypointManager.normalizeIcon(waypoint.icon), transform, menu);
    }

    private static StoredWaypoint fromMarker(@Nonnull MapMarker marker, @Nonnull String worldName, @Nonnull String ownerName, @Nonnull UUID ownerUuid, boolean shared) {
        if (marker.transform == null || marker.transform.position == null) {
            return null;
        }
        return new StoredWaypoint(marker.id, marker.name != null ? marker.name : "Waypoint", WaypointManager.normalizeIcon(marker.markerImage), marker.transform.position.x, marker.transform.position.y, marker.transform.position.z, worldName, shared, ownerUuid.toString(), ownerName);
    }

    private static String normalizeIcon(@Nullable String icon) {
        if (icon == null || icon.isEmpty()) {
            return "Coordinate.png";
        }
        if (icon.endsWith(".png")) {
            return icon;
        }
        return icon + ".png";
    }

    public static boolean isGlobalId(@Nonnull String id) {
        return id.startsWith(GLOBAL_ID_PREFIX);
    }

    private static String stripColorTags(String input) {
        if (input == null) {
            return "";
        }
        return input.replaceAll("<[^>]*>", "");
    }

    public static boolean isTrackedWorld(@Nullable World world) {
        return ExplorationEventListener.isTrackedWorld(world);
    }

    static {
        loadedPlayers = ConcurrentHashMap.newKeySet();
    }

    private static final class WaypointPersistence {
        private final Gson gson = new GsonBuilder().setPrettyPrinting().create();
        private final Path dataRoot;
        private final Path globalFile;

        WaypointPersistence(@Nonnull Path baseDir) {
            this.dataRoot = baseDir.resolve("data");
            this.globalFile = this.dataRoot.resolve("global-pings.json");
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        List<StoredWaypoint> loadPlayer(@Nonnull UUID playerUuid, @Nonnull String playerName, @Nonnull String worldName) {
            try {
                Path dir = this.dataRoot.resolve(worldName);
                if (!Files.exists(dir, new LinkOption[0])) {
                    return Collections.emptyList();
                }
                Path file = dir.resolve(String.valueOf(playerUuid) + "-pings.json");
                if (!Files.exists(file, new LinkOption[0])) {
                    return Collections.emptyList();
                }
                try (BufferedReader reader = Files.newBufferedReader(file);){
                    PlayerWaypointFile data = (PlayerWaypointFile)this.gson.fromJson((Reader)reader, PlayerWaypointFile.class);
                    if (data == null || data.waypoints == null) {
                        List<StoredWaypoint> list = Collections.emptyList();
                        return list;
                    }
                    ArrayList<StoredWaypoint> arrayList = new ArrayList<StoredWaypoint>(Arrays.asList(data.waypoints));
                    return arrayList;
                }
            }
            catch (Exception e) {
                LOGGER.warning("Failed to load waypoints for " + playerName + ": " + e.getMessage());
                return Collections.emptyList();
            }
        }

        void savePlayer(@Nonnull UUID playerUuid, @Nonnull String playerName, @Nonnull String worldName, @Nonnull List<StoredWaypoint> waypoints) {
            try {
                Path dir = this.dataRoot.resolve(worldName);
                Files.createDirectories(dir, new FileAttribute[0]);
                Path file = dir.resolve(String.valueOf(playerUuid) + "-pings.json");
                PlayerWaypointFile data = new PlayerWaypointFile(playerUuid.toString(), playerName, waypoints.toArray(new StoredWaypoint[0]));
                try (BufferedWriter writer = Files.newBufferedWriter(file, new OpenOption[0]);){
                    this.gson.toJson((Object)data, (Appendable)writer);
                }
            }
            catch (IOException e) {
                LOGGER.warning("Failed to save waypoints for " + playerName + ": " + e.getMessage());
            }
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        List<StoredWaypoint> loadGlobal() {
            try {
                if (!Files.exists(this.globalFile, new LinkOption[0])) {
                    return new ArrayList<StoredWaypoint>();
                }
                try (BufferedReader reader = Files.newBufferedReader(this.globalFile);){
                    GlobalWaypointFile data = (GlobalWaypointFile)this.gson.fromJson((Reader)reader, GlobalWaypointFile.class);
                    if (data == null || data.waypoints == null) {
                        ArrayList<StoredWaypoint> arrayList2 = new ArrayList<StoredWaypoint>();
                        return arrayList2;
                    }
                    ArrayList<StoredWaypoint> arrayList = new ArrayList<StoredWaypoint>(Arrays.asList(data.waypoints));
                    return arrayList;
                }
            }
            catch (Exception e) {
                LOGGER.warning("Failed to load global waypoints: " + e.getMessage());
                return new ArrayList<StoredWaypoint>();
            }
        }

        void saveGlobal(@Nonnull List<StoredWaypoint> waypoints) {
            try {
                Files.createDirectories(this.dataRoot, new FileAttribute[0]);
                GlobalWaypointFile data = new GlobalWaypointFile(waypoints.toArray(new StoredWaypoint[0]));
                try (BufferedWriter writer = Files.newBufferedWriter(this.globalFile, new OpenOption[0]);){
                    this.gson.toJson((Object)data, (Appendable)writer);
                }
            }
            catch (IOException e) {
                LOGGER.warning("Failed to save global waypoints: " + e.getMessage());
            }
        }
    }

    private static final class StoredWaypoint {
        @SerializedName(value="Id")
        private final String id;
        @SerializedName(value="Name")
        private final String name;
        @SerializedName(value="Icon")
        private final String icon;
        @SerializedName(value="X")
        private final double x;
        @SerializedName(value="Y")
        private final double y;
        @SerializedName(value="Z")
        private final double z;
        @SerializedName(value="World")
        private final String world;
        @SerializedName(value="Shared")
        private final boolean shared;
        @SerializedName(value="OwnerUuid")
        private final String ownerUuid;
        @SerializedName(value="OwnerName")
        private final String ownerName;

        StoredWaypoint(String id, String name, String icon, double x, double y, double z, String world, boolean shared, String ownerUuid, String ownerName) {
            this.id = id;
            this.name = name;
            this.icon = icon;
            this.x = x;
            this.y = y;
            this.z = z;
            this.world = world;
            this.shared = shared;
            this.ownerUuid = ownerUuid != null ? ownerUuid.toString() : null;
            this.ownerName = ownerName;
        }
    }

    private static final class GlobalWaypointFile {
        @SerializedName(value="Waypoints")
        private final StoredWaypoint[] waypoints;

        GlobalWaypointFile(StoredWaypoint[] waypoints) {
            this.waypoints = waypoints;
        }
    }

    private static final class PlayerWaypointFile {
        @SerializedName(value="PlayerUuid")
        private final String playerUuid;
        @SerializedName(value="PlayerName")
        private final String playerName;
        @SerializedName(value="Waypoints")
        private final StoredWaypoint[] waypoints;

        PlayerWaypointFile(String playerUuid, String playerName, StoredWaypoint[] waypoints) {
            this.playerUuid = playerUuid;
            this.playerName = playerName;
            this.waypoints = waypoints;
        }
    }
}

